## ----setup04, include=FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse   = TRUE,
  comment    = "#>",
  fig.path   = "fig/function-", 
  fig.ext    = "png",
  fig.width  = 6,
  fig.height = 4,
  dpi        = 300,
  fig.retina = 2,
  fig.align  = "center",
  out.width  = "70%"
)

if (requireNamespace("cifmodeling", quietly = TRUE)) {
  library(cifmodeling)
} else {
  stop("Package 'cifmodeling' is not available during vignette build.")
}
library(nleqslv)
library(boot)
library(ggplot2)
library(ggsurvfit)
library(patchwork)

data("diabetes.complications", package = "cifmodeling")
set.seed(1)

## ----example04-1-1, fig.cap="Cumulative incidence curves per each stratification variable", warning=FALSE, message=FALSE----
data(diabetes.complications)
diabetes.complications$fruitq1 <- ifelse(
  diabetes.complications$fruitq == "Q1","Q1","Q2 to Q4"
)
cifplot(Event(t,epsilon)~fruitq+fruitq1, data=diabetes.complications, 
        outcome.type="competing-risk",
        add.conf=TRUE, add.censor.mark=FALSE, 
        add.competing.risk.mark=FALSE, panel.per.variable=TRUE)

## ----example04-1-2, fig.cap="Cumulative incidence curves with competing risk marks", warning=FALSE, message=FALSE----
output1 <- cifcurve(Event(t,epsilon)~fruitq1, data=diabetes.complications, 
                    outcome.type="competing-risk")
output2 <- extract_time_to_event(Event(t,epsilon)~fruitq1, 
                                 data=diabetes.complications, which.event="event2")
cifplot(output1, add.conf=FALSE, add.risktable=FALSE, 
        add.censor.mark=FALSE, add.competing.risk.mark=TRUE, competing.risk.time=output2, 
        label.y="CIF of diabetic retinopathy", label.x="Years from registration",
        limits.x=c(0,8))

## ----example04-1-3, fig.cap="Cumulative incidence curves with strata labels and framed style", warning=FALSE, message=FALSE----
cifplot(Event(t,epsilon)~fruitq1, data=diabetes.complications, 
        outcome.type="competing-risk", add.conf=FALSE, add.risktable=FALSE, 
        add.estimate.table=TRUE, add.censor.mark=FALSE, add.competing.risk.mark=TRUE, 
        competing.risk.time=output2, label.y="CIF of diabetic retinopathy", 
        label.x="Years from registration", limits.x=c(0,8),
        label.strata=c("High intake","Low intake"), level.strata=c("Q2 to Q4","Q1"), 
        order.strata=c("Q1", "Q2 to Q4"), style="framed")

## ----example04-1-4, fig.cap="Joint estimation of unadjusted risk ratios at 8 years using polyreg()", warning=FALSE, message=FALSE----
output3 <- polyreg(nuisance.model=Event(t,epsilon)~1, exposure="fruitq1", 
          data=diabetes.complications, effect.measure1="RR", effect.measure2="RR", 
          time.point=8, outcome.type="competing-risk", 
          report.nuisance.parameter=TRUE)
coef(output3)
vcov(output3)
summary(output3)

## ----example04-2-1, fig.cap="Survival curves from cifplot()", warning=FALSE, message=FALSE----
diabetes.complications$d <- as.integer(diabetes.complications$epsilon>0)
cifplot(Event(t,d) ~ fruitq1, data=diabetes.complications, 
outcome.type="survival", add.conf=TRUE, add.censor.mark=FALSE, 
add.competing.risk.mark=FALSE, label.y="Survival (no complications)", 
label.x="Years from registration", label.strata=c("High intake","Low intake"),
level.strata=c("Q2 to Q4","Q1"), order.strata=c("Q1", "Q2 to Q4"))

## ----example04-2-2, fig.cap="Cause-specific estimation of an unadjusted risk ratio at 8 years using polyreg()", warning=FALSE, message=FALSE----
output4 <- polyreg(nuisance.model=Event(t,d)~1, 
          exposure="fruitq1", strata="strata", data=diabetes.complications,
          effect.measure1="RR", time.point=8, outcome.type="survival")
summary(output4)

## ----example04-3, fig.cap="Joint estimation of adjusted risk ratios at 8 years using polyreg()", warning=FALSE, message=FALSE----
output5 <- polyreg(nuisance.model=Event(t,epsilon)~age+sex+bmi+hba1c
          +diabetes_duration+drug_oha+drug_insulin+sbp+ldl+hdl+tg
          +current_smoker+alcohol_drinker+ltpa, 
          exposure="fruitq1", strata="strata", data=diabetes.complications,
          effect.measure1="RR", time.point=8, outcome.type="competing-risk")
summary(output5)

## ----example04-4, fig.cap="Cumulative incidence curves for event 1 vs event 2 using cifpanel()", warning=FALSE, message=FALSE----
output6 <- cifpanel(
 rows.columns.panel = c(1,2),
 formula            = Event(t, epsilon) ~ fruitq1,
 data               = diabetes.complications,
 outcome.type       = "competing-risk",
 code.events        = list(c(1,2,0), c(2,1,0)),
 label.y            = c("CIF of diabetic retinopathy", "CIF of macrovascular complications"),
 label.x            = "Years from registration",
 label.strata       = list(c("High intake","Low intake")),
 title.plot         = c("Diabetic retinopathy", "Macrovascular complications"),
 legend.position    = "bottom",
 legend.collect     = TRUE
)
print(output6)

